<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(200); exit; }

// ---- Получение реального IP (Cloudflare → X-Forwarded-For → REMOTE_ADDR) ----
function getClientIP() {
    // Cloudflare всегда пишет реальный IP сюда — главный приоритет
    if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
        $ip = trim($_SERVER['HTTP_CF_CONNECTING_IP']);
        if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
    }
    // X-Real-IP (nginx proxy)
    if (!empty($_SERVER['HTTP_X_REAL_IP'])) {
        $ip = trim($_SERVER['HTTP_X_REAL_IP']);
        if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
    }
    // X-Forwarded-For — берём ТОЛЬКО первый IP (клиентский)
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = trim(explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0]);
        if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
    }
    return $_SERVER['REMOTE_ADDR'] ?? 'UNKNOWN';
}

// ---- Геолокация через ip-api.com (curl) ----
function getGeoInfo($ip) {
    $default = [
        'country'     => 'Unknown',
        'countryCode' => 'XX',
        'city'        => 'Unknown',
        'region'      => 'Unknown',
        'isp'         => 'Unknown'
    ];

    // Не запрашиваем геолокацию для приватных/Cloudflare IP
    if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        $default['isp'] = 'Private/Local IP';
        return $default;
    }

    $url = "http://ip-api.com/json/{$ip}?fields=status,message,country,countryCode,regionName,city,isp,org&lang=en";

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 5,
        CURLOPT_CONNECTTIMEOUT => 3,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT      => 'Mozilla/5.0',
    ]);
    $response = curl_exec($ch);
    $curlError = curl_error($ch);
    curl_close($ch);

    if (!$response || $curlError) {
        // Fallback: попробуем через file_get_contents если curl не сработал
        $response = @file_get_contents($url);
    }

    if (!$response) return $default;

    $json = json_decode($response, true);
    if (!$json || $json['status'] !== 'success') return $default;

    return [
        'country'     => $json['country']    ?? 'Unknown',
        'countryCode' => $json['countryCode'] ?? 'XX',
        'city'        => $json['city']        ?? 'Unknown',
        'region'      => $json['regionName']  ?? 'Unknown',
        'isp'         => $json['isp']         ?? ($json['org'] ?? 'Unknown'),
    ];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $raw  = file_get_contents('php://input');
    $data = json_decode($raw, true) ?: [];

    $ip  = getClientIP();
    $geo = getGeoInfo($ip);

    $entry = [
        'datetime'         => date('Y-m-d H:i:s'),
        'ip'               => $ip,   // <- теперь всегда ОДИН чистый IP
        'country'          => $geo['country'],
        'countryCode'      => $geo['countryCode'],
        'city'             => $geo['city'],
        'region'           => $geo['region'],
        'isp'              => $geo['isp'],
        'event'            => $data['event']            ?? 'unknown',
        'sessionId'        => $data['sessionId']        ?? 'N/A',
        'timestamp'        => $data['timestamp']        ?? date('c'),
        'userAgent'        => $data['userAgent']        ?? '',
        'screenResolution' => $data['screenResolution'] ?? '',
        'language'         => $data['language']         ?? '',
        'referrer'         => $data['referrer']         ?? '',
        'currentUrl'       => $data['currentUrl']       ?? '',
    ];

    if (!is_dir('logs')) mkdir('logs', 0755, true);

    $file = 'logs/events_' . date('Y-m-d') . '.json';
    $logs = file_exists($file)
        ? (json_decode(file_get_contents($file), true) ?: [])
        : [];

    $logs[] = $entry;
    file_put_contents($file, json_encode($logs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

    echo json_encode(['status' => 'success', 'ip' => $ip, 'geo' => $geo]);
} else {
    http_response_code(405);
}
?>